<?php declare(strict_types=1);

namespace Shopware\Core\Maintenance\Staging\Command;

use Psr\EventDispatcher\EventDispatcherInterface;
use Shopware\Core\Framework\Adapter\Console\ShopwareStyle;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\Log\Package;
use Shopware\Core\Maintenance\Staging\Event\SetupStagingEvent;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * @internal
 *
 * @phpstan-import-type DomainRewriteRule from SetupStagingEvent
 */
#[AsCommand(
    name: 'system:setup:staging',
    description: 'Installs the Shopware 6 system in staging mode',
)]
#[Package('framework')]
class SystemSetupStagingCommand extends Command
{
    /**
     * @param list<DomainRewriteRule> $domainMappings
     */
    public function __construct(
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly SystemConfigService $systemConfigService,
        public readonly bool $disableMailDelivery,
        public readonly array $domainMappings,
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this->addOption('force', 'f', InputOption::VALUE_NONE, 'Force setup of staging system');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new ShopwareStyle($input, $output);

        if (!$input->getOption('force') && !$io->confirm('This command will install the Shopware 6 system in staging mode. It will overwrite existing data in this database, make sure you use a staging database and have a backup', false)) {
            return self::FAILURE;
        }

        $event = new SetupStagingEvent(
            Context::createCLIContext(),
            $io,
            $this->disableMailDelivery,
            $this->domainMappings
        );
        $this->eventDispatcher->dispatch($event);

        $this->systemConfigService->set(SetupStagingEvent::CONFIG_FLAG, true);

        return $event->canceled ? self::FAILURE : self::SUCCESS;
    }
}
